<?php

namespace App\Services\Settlement;

use App\Domicilio;
use App\DomicilioHasSettlement;
use App\LogSender;
use App\Services\Domicilio\DomicilioServices;
use App\Settlement;
use Barryvdh\DomPDF\Facade as PDF;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use App\Traits\ApiResponser;
use App\Enums\DomicilioStatusConst;
use App\Http\Resources\Settlements\SettlementResource;
use App\Services\Parameters\ParameterAccesorService;

class SettlementService
{
    use ApiResponser;

    protected $domiciliosService;
    protected $calculationService;

    public function __construct(
        DomicilioServices $domicilioServices,
        SettlementCalculationService $calculationService
    ) {
        $this->domiciliosService = $domicilioServices;
        $this->calculationService = $calculationService;
    }

    public function listDomiciliosSettlements(
        int $perPage,
        string $from,
        string $to,
        string $search,
        ?int $senderId = null
    ) {

        try {
            $query = Settlement::query()
                ->with('createdBy', 'domicilios.logSender', 'domicilios.dropshipper', 'domicilios.Collections') // Incluye la relación con dropshipper.
                ->when($search, function ($query) use ($search) {
                    $query->where(function ($q) use ($search) {
                        $q->where('id', 'like', "%{$search}%")
                            ->orWhere('pdf_url', 'like', "%{$search}%");
                    });
                })
                ->when($from && $to, function ($query) use ($from, $to) {
                    $query->whereBetween('created_at', [$from, $to]);
                })
                ->when($senderId, function ($query) use ($senderId) {
                    $query->whereHas('domicilios', function ($q) use ($senderId) {
                        $q->where('log_sender_id', $senderId);
                    });
                })
                ->orderBy('created_at', 'DESC');

            return SettlementResource::collection($query->paginate($perPage));
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    public function getSettlementInformation(int $id)
    {
        $settlement = Settlement::findOrFail($id);
        $dataSettlement = $this->generatePdfData($settlement, $settlement->domicilios);

        return $dataSettlement;
    }

    /**
     * @throws \Exception
     */
    public function getPDFByDomicilio(int $id, $domicilios, $senderId = null)
    {
        $settlement = Settlement::findOrFail($id);
        $sender = null;

        if ($senderId) {
            $sender = LogSender::findOrFail($senderId);
        }

        $pdfUrl = $this->emitPDF($settlement, $domicilios, $sender);
        $settlement->pdf_url = $pdfUrl;
        $settlement->is_open = false;
        $settlement->save();

        return $settlement->pdf_url;
    }

    /*
     * METODO PARA MARCAR DOMICILIOS COMO LIQUIDOS
     * Y REGISTRARLOS EN SETTLEMENTS PARA AGRUPARLOS
     */
    public function changeDomiciliosToSettlement($items)
    {
        if (!isset($items) || count($items) === 0) {
            throw new BadRequestException("No hay un array correcto de domicilios");
        }

        $settlement = Settlement::create([
            "created_by" => Auth::id(),
            "modified_by" => Auth::id()
        ]);

        foreach ($items as $item) {
            $this->domiciliosService->changeStatusPaid($item["domicilio_id"], true);
            DomicilioHasSettlement::create([
                "domicilio_id" => $item["domicilio_id"],
                "settlement_id" => $settlement->id
            ]);
        }

        $settlement->save();

        return $settlement;
    }

    public function emitPDF($settlement, $domicilios, $sender = null)
    {
        $configuration = \App\Configuration::all()->first();

        $pdfData = $this->generatePdfData($settlement, $domicilios, $sender);

        $domicilios = collect($pdfData->domicilios);

        $comission = ParameterAccesorService::getSettlementCollectionComission();
        // Calcular los totales
        $totalCollectionValue = $domicilios->sum('collection_value');
        $totalCollectionCommission = $domicilios->sum('collection_comission');
        //$totalFullfilment = $domicilios->sum('fullfilment');
        $totalFullfilment = ParameterAccesorService::isActivedFullfillment() ? $domicilios->sum('fullfilment') : 0;
        $totalLastMille = $domicilios->sum('last_mille');
        $totalTotal = $domicilios->sum('total');
        $isFullfillment = ParameterAccesorService::isActivedFullfillment();
        $isComission = ParameterAccesorService::isActivedSettlementCollectionComission();

        $totalOperationCost = isset($pdfData->has_dropshipper) && $pdfData->has_dropshipper ? $this->calculationService->sumOperationCost($pdfData) : 0;

        if (!ParameterAccesorService::isActivedFullfillment()) {
            foreach ($pdfData->domicilios as $domicilio) {
                $domicilio->fullfilment = 0;
            }
        }
        // Pasar los totales a la vista
        $pdf = PDF::loadView('onx.settlement', compact(
            'pdfData',
            'configuration',
            'totalCollectionValue',
            'totalCollectionCommission',
            'totalFullfilment',
            'totalLastMille',
            'totalTotal',
            'totalOperationCost',
            'comission',
            'isFullfillment',
            'isComission'
        ))->setPaper('a4', 'landscape');

        $path = public_path('pdf/');
        $fileName =  'settlement_' . $settlement->id . '.pdf';

        // Crear la carpeta si no existe
        if (!file_exists($path)) {
            mkdir($path, 0777, true); // 0777 es el permiso para que el directorio sea legible/escribible/ejecutable
        }

        $pdf->save($path . '/' . $fileName);
        $file = public_path() . "/pdf/" . $fileName;
        if (ENV('APP_ENV') == 'production') {
            $url = url('pdf');
        } else {
            $url = url('pdf');
        }
        if (strpos(url('/'), 'ecomdex') !== false) {
            $url = url('api/public/pdf');
        }
        return "{$url}/{$fileName}";
    }

    public function removeSettlementDomicilio(int $id)
    {
        $settlement = Settlement::findOrFail($id);
        $domicilios = $settlement->domicilios;

        foreach ($domicilios as $domicilio) {
            $domicilio->liquidated = 0;
            $domicilio->save();
            DomicilioHasSettlement::where("domicilio_id", $domicilio->id)
                ->delete();
        }

        $settlement->delete();

        return $settlement;
    }

    public function removeDomicilioInSettlement(int $id)
    {
        $domicilio = Domicilio::findOrFail($id);
        $relation = DomicilioHasSettlement::where("domicilio_id", $domicilio->id)->first();
        $settlement = $relation->settlement;
        $relation->delete();

        if ($settlement->domicilios->count() == 0) {
            $settlement->delete();
        }

        $domicilio->liquidated = 0;
        $domicilio->save();

        return $domicilio;
    }

    private function isThereDropshipperInformation($pdfData)
    {

        foreach ($pdfData->domicilios as $domicilio) {
            if (isset($domicilio->name_dropshipper)) {
                return true;
            }
        }

        return false;
    }

    private function generatePdfData($settlement, $domicilios, $sender = null)
    {
        $pdfData = new \stdClass();
        $domiciliosArray = array();
        $pdfData->id = $settlement->id;

        if ($sender) {
            $pdfData->sender = $sender->name;
        }

        foreach ($domicilios as $domicilio) {
            $domicilio = Domicilio::findOrFail($domicilio["id"]);
            $obj = $this->calculationService->generateDomicilioData($domicilio);
            array_push($domiciliosArray, $obj);
        }

        $pdfData->domicilios = $domiciliosArray;
        if ($this->isThereDropshipperInformation($pdfData)) {
            $pdfData->has_dropshipper = true;
        }
        $pdfData->created_at = $settlement->created_at;
        $pdfData->totalDomicilios = count($domiciliosArray);

        return $pdfData;
    }

    public function addDomicilioToSettlement(int $settlementId, string $reference)
    {
        $domicilio = Domicilio::where("reference", $reference)->get()->first();
        $settlement = Settlement::findOrFail($settlementId);
        $domicilioHasSettlement = DomicilioHasSettlement::where("domicilio_id", $domicilio->id)
            ->get()->first();

        if (!$domicilio) {
            throw new NotFoundHttpException("No existe el domicilio con referencia $reference");
        }

        if ($domicilio->status != DomicilioStatusConst::VERIFIED) {
            throw new BadRequestException("El domicilio $reference no se encuentra verificado");
        }

        if (!$settlement->is_open) {
            throw new BadRequestException("La liquidacion $settlementId ya se encuentra cerrada");
        }

        if ($domicilioHasSettlement) {
            throw new BadRequestException("El domicilio $reference ya se encuentra en otra liquidacion");
        }

        return DomicilioHasSettlement::create([
            "domicilio_id" => $domicilio->id,
            "settlement_id" => $settlementId
        ]);
    }
}
