<?php

namespace App\Services\Domicilio;

use App\Client;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use PDF;
use App\Evidence;
use Carbon\Carbon;
use App\Domicilio;
use Dompdf\Options;
use App\Helpers\Helper;
use Illuminate\Support\Str;
use App\Traits\ApiResponser;
use App\Exports\DomicilioExcel;
use App\Imports\DomicilioImport;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Controllers\ClientController;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Services\LogSender\LogSenderServices;
use App\Http\Resources\Domicilio\DomicilioResource;
use App\Http\Resources\Domicilio\DomicilioDataResource;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\DropshipperSettlement;
use App\Enums\DomicilioStatusConst;
use App\Services\Onexfy\OnexfyService;
use Illuminate\Http\Request;
use App\DomiciliosRescheduling;
use App\CoordinatorHasCity;
use App\Services\Parameters\ParameterAccesorService;

/**
 *
 */
class DomicilioServices
{
    use ApiResponser;

    protected $senderServices;
    protected $onexfyService;

    public function __construct(LogSenderServices $logSenderServices, OnexfyService $onexfyService)
    {
        $this->senderServices = $logSenderServices;
        $this->onexfyService = $onexfyService;
    }

    public function getDomiciliosForUser(Request $request)
    {
        $perPage = isset($request->per_page) ? $request->per_page : (isset($request->rowsPerPage) ? $request->rowsPerPage : 100);
        $from = $request->from ? $request->from . ' 00:00:00' : Carbon::now()->startOfYear();
        $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfYear();
        $clientId = $request->client_id ?? null;

        // select query
        $domicilios = Domicilio::select([
            'domicilios.id as id',
            'domicilios.reference as reference',
            'domicilios.total as price',
            'domicilios.status as status',
            'domicilios.type as service',
            'domicilios.weight as weight',
            'domicilios.delivery as delivery',
            'domicilios.log_sender_id as sender_id',
            'domicilios.liquidated as liquidated',
            'domicilios.content as content',
            DB::raw(
                "(SELECT domicilios_reschedulings.rescheduling_date FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_date"
            ),
            DB::raw(
                "(SELECT domicilios_reschedulings.rescheduling_start_hour FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_start_hour"
            ),
            DB::raw(
                "(SELECT domicilios_reschedulings.rescheduling_end_hour FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_end_hour"
            ),
            DB::raw(
                "(SELECT domicilios_reschedulings.description FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_description"
            ),
            DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id LIMIT 1) as city"),
            DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id LIMIT 1) as zone"),
            DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id LIMIT 1) as client"),
            DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
            'domicilios.created_at',
            'domicilios.updated_at',
            DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
            DB::raw("(SELECT SUM(value) FROM domicilio_collections WHERE domicilio_collections.domicilio_id = domicilios.id) as collection"),
            DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
            DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
            DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
            DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")
        ])
            ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
            ->orderBy('domicilios.id', 'DESC')
            ->where('domicilios.log_sender_id', 'LIKE', '%' . $request->sender_id . '%');

        if (is_null($clientId) && is_null($request->search)) {
            $domicilios->whereBetween('domicilios.created_at', [$from, $to]);
        }
        // Filtro por client_id
        if (!is_null($clientId)) {
            $domicilios->where('domicilios.client_id', '=', $clientId);
        }

        if (!is_null($request->status)) {
            $domicilios->where('domicilios.status', '=', $request->status);
        }

        if (!is_null($request->domiciliario_id)) {
            $domicilios->where('domicilios.domiciliario_id', 'LIKE', '%' . $request->domiciliario_id . '%');
        }
        if (!is_null($request->liquidated)) {
            $domicilios->where('domicilios.liquidated', 'LIKE', '%' . $request->liquidated . '%');
        }
        if (!is_null($request->city_id)) {
            $domicilios->where('domicilios.city_id', '=', $request->city_id);
        }
        if (!is_null($request->zone_id)) {
            $domicilios->where('domicilios.zone_id', '=', $request->zone_id);
        }

        $columns = [
            'service',
            'client',
            'domiciliario',
            'id',
            'reference',
            'weight',
            'status',
            'sender',
            'client_city',
            'client_zone',
            'rescheduling_date'
        ];

        if (isset($request->search)) {
            $domicilios = $domicilios->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $request->search . "%'");
        }

        if ($request->user() && is_numeric($request->user()->domiciliario_id)) {
            $domicilios = $domicilios->where('domiciliario_id', $request->user()->domiciliario_id);
        }
        if ($request->user() && is_numeric($request->user()->log_sender_id)) {
            $domicilios = $domicilios->where('log_sender_id', $request->user()->log_sender_id);
        }
        if ($request->user() && $request->user()["role_id"] == 3) {
            $user = $request->user();
            $cityIds = CoordinatorHasCity::where('user_id', $user->id)->pluck('city_id');
            $domicilios->whereIn('domicilios.city_id', $cityIds);
        }

        if (!is_null($request->status)) {
            $domicilios->orderBy('domicilios.updated_at', 'DESC');
        }

        $domicilios = $domicilios->paginate($perPage);
        return $domicilios;
    }


    //filter by ref domicilio
    public function filterByRef($reference, $mainId)
    {
        $issetDomicilio = Domicilio::where('reference', '=', $reference)->where('user_id', '=', $mainId)->first();
        if (isset($issetDomicilio)) {
            return new DomicilioResource($issetDomicilio);
        } else {
            return false;
        }
    }

    /**
     *  change status in bulk
     * @param array $items
     * @param string $status
     */
    public function changeStatusOrder(
        array $items,
        string $status,
        string $observation = '',
        $date = null,
        $startTime = null,
        $endTime = null
    ) {
        try {
            //$updateDomicilio = Domicilio::whereIn('id', $items)->update(['status' => $status]);
            $this->setStatuses($items, $status, $observation, $date, $startTime, $endTime);
            return $this->successResponse([
                'success' => true,
                //'data' => $updateDomicilio,
                'message' => 'Se ha cambiado el estado de los domicilios seleccionados.'
            ]);
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     *  change status liqueidado
     * @param int $id
     * @param boolean $status_paid
     */
    public function changeStatusPaid($id, $status_paid)
    {
        try {
            $updateDomicilio = Domicilio::where('id', $id)->update(['paid' => $status_paid, 'liquidated' => $status_paid]);
            return $this->successResponse([
                'success' => true,
                'data' => $updateDomicilio,
                'message' => 'Se ha cambiado el estado de liquidado'
            ]);
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * set statuses
     * @param array $items
     * @param string $status
     */

    public function setStatuses(
        $items,
        $statusCode,
        $observation = ''
    ) {
        foreach ($items as $dom => $domicilio) {
            $domicilio = Domicilio::findOrFail($domicilio);

            /*
            if (
                $domicilio->status === DomicilioStatusConst::DELIVERED
                || $domicilio->status === DomicilioStatusConst::VERIFIED
            ) {
                if ($statusCode !== DomicilioStatusConst::REVERSED && $statusCode !== DomicilioStatusConst::VERIFIED) {
                    return "Solo se puede cambiar a estado Reversado o Verificado";
                }
            } */

            // Si el nuevo estado es "Entregado", marcar como pagado
            if ($statusCode === DomicilioStatusConst::DELIVERED) {
                $domicilio->paid = 1;
                //$domicilio->update();
            }

            $domicilio->status = $statusCode;
            $domicilio->save();


            // Crear el nuevo estado
            $status = \App\Status::create([
                'status' => $statusCode,
                'observation' => $observation,
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
                'updated_by' => Auth::id(),
            ]);

            // Notificar cambio de estado
            if ($statusCode === DomicilioStatusConst::NEW_EVENT) {
                $this->onexfyService->notifyChangeStatusOrder(
                    $domicilio->reference,
                    $statusCode,
                    $observation
                );
            } elseif ($statusCode === DomicilioStatusConst::RETURNED) {
                $this->onexfyService->notifyChangeStatusOrder(
                    $domicilio->reference,
                    $statusCode,
                    $observation,
                    $domicilio->content
                );
            } else {
                $this->onexfyService->notifyChangeStatusOrder(
                    $domicilio->reference,
                    $statusCode,
                    "Cambio de estatus manual"
                );
            }
        }
    }


    /**
     * Get status from order id
     * @param int $orderId
     */
    public function getStatusByOrderId(int $orderId)
    {
        try {
            $statuses = DB::table('statuses')
                ->select(
                    'statuses.status as status',
                    DB::raw("DATE_FORMAT(statuses.created_at, '%d/%m/%Y') as date"),
                    DB::raw("DATE_FORMAT(statuses.created_at, '%h:%i:%s %p') as hour")
                )
                ->where('statuses.domicilio_id', $orderId)
                ->get();
            return $statuses;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * upload evidence
     * @param File $file
     * @param int $domicilioId
     * @param string $type
     */
    public function uploadEvidence($file, $domicilioId, $type = 'Entrega')
    {
        try {
            if (isset($file)) {
                $path = Helper::uploadImage($file, 'evidence');

                //Guardar la ruta relativa en la base de datos
                $relativePath = 'images/evidence/' . $path;

                //Crear el registro de evidencia
                $evidence = Evidence::create([
                    'domicilio_id' => $domicilioId,
                    'path' => $relativePath, // Guardamos ruta relativa
                    'type' => $type,
                ]);

                //Para la respuesta, construimos la URL completa
                $evidence->full_url = url($relativePath);

                return $evidence;
            }
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Store new order
     * @param array $request
     */
    public function storeNewOrder($request)
    {
        try {
            // generate client or update
            if (!env('ONEX_DELIVERY_PREFIX') || env('ONEX_DELIVERY_PREFIX') == "") {
                return $this->errorUnprocessableEntityResponse(
                    "Debe haber un prefijo para crear la referencia"
                );
            }
            $clientId = null;
            if (isset($request['client']) and isset($request['client']['id'])) {
                $clientId = $request['client']['id'];
                $clientId = ClientController::saveNewClient($request['client']);
            } else {
                $clientId = ClientController::saveNewClient($request['client']);
            }
            $request['zone_id'] = isset($request['client']['zone_id']) ? $request['client']['zone_id'] : null;
            $request['city_id'] = isset($request['client']['city_id']) ? $request['client']['city_id'] : null;
            $domicilio = Domicilio::create(array_merge($request, ['status' => 'Pendiente', 'client_id' => $clientId]));
            //$reference = Str::padLeft($domicilio->id, 9, 0);
            $numberGuia = Str::padLeft($domicilio->id, 9, 0);
            $reference = env('ONEX_DELIVERY_PREFIX') . "-$numberGuia";
            $domicilio->update(['reference' => $reference]);
            // generate initial status
            $status = \App\Status::create([
                'status' => 'Pendiente',
                'observation' => isset($request['observation']) ? $request['observation'] : '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
            // save products in orders
            foreach ($request['products'] as $prods => $product) {
                \App\DomicilioProduct::create([
                    'reference' => isset($product['reference']) ? $product['reference'] : '',
                    'name' => isset($product['name']) ? $product['name'] : '',
                    'price' => isset($product['price']) ? $product['price'] : '',
                    'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
                    'domicilio_id' => $domicilio->id
                ]);
            }
            // return order
            return $domicilio;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * update domicilio
     * @param array $request
     */
    public function updateOrder($request)
    {
        try {
            // Encuentra el domicilio
            $domicilio = Domicilio::findOrFail($request['id']);
            $request['zone_id'] = isset($request['client']['zone_id']) ? $request['client']['zone_id'] : null;
            $request['city_id'] = isset($request['client']['city_id']) ? $request['client']['city_id'] : null;

            // Obtener datos del cliente
            $clientId = null;
            if (isset($request['client']) && isset($request['client']['id'])) {
                $clientId = $request['client']['id'];
                $clientId = ClientController::saveNewClient($request['client']);
            } else {
                $clientId = ClientController::saveNewClient($request['client']);
            }

            if (isset($request['client'])) {
                $clientId = $request['client']['id'];
                $client = Client::findOrFail($clientId);
                $client->update($request["client"]);
            }

            // Obtener productos
            $products = $request['products'];

            // Eliminar datos no necesarios
            unset($request['client']);
            unset($request['products']);
            $domicilio->liquidated = isset($request['liquidated']) ? 1 : 0;
            $domicilio->content = isset($request['content']) ? $request['content'] : '';
            $domicilio->update($request);

            // Validar y actualizar productos
            foreach ($products as $product) {
                if (isset($product['id'])) {
                    $productBd = \App\DomicilioProduct::findOrFail($product['id']);
                    $productBd->update($product);
                } else {
                    \App\DomicilioProduct::create([
                        'reference' => isset($product['reference']) ? $product['reference'] : '',
                        'name' => isset($product['name']) ? $product['name'] : '',
                        'price' => isset($product['price']) ? $product['price'] : '',
                        'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
                        'domicilio_id' => $domicilio->id
                    ]);
                }
            }

            // Generar la guía después de actualizar la orden
            //$this->generateGuide($domicilio->reference);

            // Retornar el domicilio actualizado
            return $domicilio;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * update domiciliary
     * @param array $request
     */
    public function updateDomiciliary($request)
    {
        try {
            // find domicilios
            $domicilio = Domicilio::findOrFail($request['id']);
            $domicilio->domiciliario_id = $request['domiciliary_id'];
            $domicilio->update();
            return $domicilio;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }


    public function assignDomiciliary($request)
    {
        try {
            $domicilio = Domicilio::findOrFail($request['id']);
            $domicilio->domiciliario_id = $request['domiciliary_id'];
            $domicilio->update();
            return $domicilio;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function assignDomiciliaryApk($request)
    {
        try {
            $reference = $request['reference'];
            $domiciliaryId = $request['domiciliary_id'];

            $domicilio = Domicilio::where('reference', $reference)->firstOrFail();

            $domicilio->domiciliario_id = $domiciliaryId;
            $domicilio->save();
            return response()->json($domicilio);
        } catch (ModelNotFoundException $e) {
            return $this->handlerException('No se encontró ningún domicilio con la referencia proporcionada.', 404);
        } catch (Exception $e) {
            return $this->handlerException('Error al procesar la solicitud: ' . $e->getMessage(), 400);
        }
    }

    public function getLastOrders($user, $status = null)
    {
        try {
            if (!isset($status)) {
                $status = DomicilioStatusConst::PENDING;
            }

            $fields = [
                'domicilios.id',
                'domicilios.status as status',
                'domicilios.reference as reference',
                'domicilios.total as price',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT address FROM clients WHERE id = domicilios.client_id) as address"),
                'domicilios.updated_at as updated_at'
            ];

            $query = Domicilio::select($fields)
                ->with('Collections')
                ->with(['reschedules' => function ($query) {
                    $query->whereDate('rescheduling_date', Carbon::today());
                }])
                ->when($user['role_id'] === 4, function ($query) use ($user) {
                    return $query->where('log_sender_id', $user['log_sender_id']);
                });

            $statusGroups = [
                DomicilioStatusConst::ON_THE_ROAD => [
                    DomicilioStatusConst::ON_THE_ROAD,
                    DomicilioStatusConst::IN_WAREHOUSE,
                    DomicilioStatusConst::ON_THE_WAY
                ],
                DomicilioStatusConst::DELIVERED => [
                    DomicilioStatusConst::DELIVERED,
                    DomicilioStatusConst::VERIFIED
                ],
                DomicilioStatusConst::NEW_EVENT => [
                    DomicilioStatusConst::NEW_EVENT,
                    DomicilioStatusConst::RETURNED
                ]
            ];

            // Aplicar el filtro de estados
            if (array_key_exists($status, $statusGroups)) {
                $query->whereIn('domicilios.status', $statusGroups[$status]);
            } else {
                $query->where('domicilios.status', $status);
            }

            // Manejo especial para domicilios reprogramados
            if ($status === DomicilioStatusConst::RESCHEDULED) {
                $query->whereHas('reschedules', function ($q) {
                    $q->whereDate('rescheduling_date', Carbon::today());
                })
                    ->orderBy(
                        DomiciliosRescheduling::select('rescheduling_date')
                            ->whereColumn('domicilio_id', 'domicilios.id')
                            ->whereDate('rescheduling_date', Carbon::today())
                            ->latest()
                            ->take(1)
                    );
            } else {
                $query->where('domicilios.updated_at', '>=', Carbon::now()->subHours(45))
                    ->orderBy('domicilios.id', 'DESC');
            }

            $domicilios = $query->get()->take(10);

            $domicilios->each(function ($domicilio) {
                $domicilio->collection_values_count = $domicilio->collection_values_count;

                // Para domicilios reprogramados, añadir la fecha de reprogramación
                if ($domicilio->status === DomicilioStatusConst::RESCHEDULED && $domicilio->reschedules->isNotEmpty()) {
                    $domicilio->rescheduling_date = $domicilio->reschedules->first()->rescheduling_date;
                }
            });

            return $domicilios;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /*
    public function getLastOrders($user)
    {
        try {
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as original_delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id) as zone"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id) as client"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone"),
                // Aplicar descuento al campo delivery
                DB::raw("(domicilios.delivery - (domicilios.delivery *
                         (SELECT COALESCE(discount, 0) FROM log_senders WHERE id = domicilios.log_sender_id) / 100)) as delivery")
            ])
                ->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'domicilios.domiciliario_id')
                ->orderBy('domicilios.id', 'DESC')
                ->when($user['role_id'] === 4, function ($query) use ($user) {
                    return $query->where('log_sender_id', $user['log_sender_id']);
                })
                ->get()
                ->take(10);

            return $domicilios;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }


    public function getLastordersAll($user)
    {
        try {
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id) as zone"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id) as client"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")
            ])
                ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
                ->orderBy('domicilios.id', 'DESC')
                // ->when($user['role_id'] === 4, function ($query) use($user) {
                //     return $query->where('log_sender_id', $user['log_sender_id']);
                // })
                ->get()
                ->take(10);
            return $domicilios;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }*/

    public function getLastordersNew($dociliciarios, $status = null)
    {


        try {
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id) as zone"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id) as client"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")
            ])
                ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
                ->where('domicilios.domiciliario_id', $dociliciarios)
                ->when($status !== null, function ($query) use ($status) {
                    return $query->where('domicilios.status', $status);
                })
                ->orderBy('domicilios.id', 'DESC')
                ->take(10)
                ->get();

            return $domicilios;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * generate order guides
     * @param string $orderReference
     */
    public function generateGuide(string $orderReference)
    {
        try {
            $domicilio = Domicilio::where('reference', $orderReference)->first();
            $fileName =  'guide_' . $domicilio->reference . '.' . 'pdf';
            $url = url('/guides');
            // prepare barcode and qr code
            $qrCode = QrCode::size(150)->generate($domicilio->reference);
            $configuration = \App\Configuration::all()->first();
            $formats = [
                '10x13' => [0, 0, 283.46, 368.5],
                '10x10' => [0, 0, 283.46, 283.46]
            ];

            $formart = '10x13';
            if (isset($configuration) && !empty($configuration->format_guide)) {
                $formart = isset($formats[$configuration->format_guide]) ? $configuration->format_guide : $formart;
            }

            // generate pdf
            $pdf = PDF::loadView('onx/guide_' . $formart, array(
                'order' => $domicilio,
                'qrCode' => $qrCode,
                'itemsLabel' =>  $this->generateItemLabel($domicilio->Products),
                'configuration' => $configuration,
            ));
            $options = new Options();
            $options->set('isPhpEnabled', true);
            $options->set('isRemoteEnabled', true);
            $pdf->getDomPDF()->setPaper($formats[$formart])->setOptions($options);
            $path = public_path('guides/');
            $pdf->save($path . '/' . $fileName);
            $file = public_path() . "guides/" . $fileName;
            $domicilio->invoice_url = "{$url}/{$fileName}";
            $domicilio->save();

            // si posee pdf lo devolvemos una vez actualizado el nuevo
            if (!empty($domicilio->invoice_url)) {
                return [
                    'file' => "{$url}/{$fileName}"
                ];
            }

            $domicilio->update([
                'status' => 'Guía Generada',
                'invoice_url' => "{$url}/{$fileName}",

            ]);
            $status = \App\Status::create([
                'status' => 'Guía Generada',
                'observation' => '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
            return [
                'file' => "{$url}/{$fileName}",
                'domicilio' => $domicilio
            ];
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function generateGuideCheck(array $orderReferences)
    {
        try {
            $guides = [];

            $configuration = \App\Configuration::all()->first();
            foreach ($orderReferences as $orderReference) {
                $domicilio = Domicilio::where('reference', $orderReference)->first();

                if (!$domicilio) {
                    throw new Exception("No se encontró el domicilio con referencia: {$orderReference}");
                }

                $fileName = 'guide_' . $domicilio->reference . '.pdf';
                $url = url('api/public/guides');

                // Verificar si el archivo PDF ya existe
                if ($domicilio->invoice_url) {
                    $guides[] = [
                        'file' => "{$url}/{$fileName}",
                        'domicilio' => $domicilio
                    ];
                    continue;
                }

                // Generar el código QR
                $qrCode = QrCode::size(150)->generate($domicilio->reference);

                // Generar el PDF de la guía
                $pdf = Pdf::loadView('onx/guides', [
                    'order' => $domicilio,
                    'qrCode' => $qrCode,
                    'configuration' => $configuration,
                    'itemsLabel' =>  $this->generateItemLabel($domicilio->Products),
                    // Agrega otros datos necesarios aquí
                ]);

                // Configuración del PDF
                $options = new Options();
                $options->set('isPhpEnabled', true);
                $options->set('isRemoteEnabled', true);
                $pdf->getDomPDF()->setPaper([0, 0, 100, 100])->setOptions($options);

                // Guardar el archivo PDF
                $path = public_path('guides/');
                $pdf->save($path . $fileName);

                // Comprobar si el archivo PDF realmente fue creado
                if (!file_exists($path . $fileName) || filesize($path . $fileName) == 0) {
                    throw new Exception("El archivo PDF {$fileName} no fue creado o está vacío.");
                }

                // Actualizar el domicilio con la URL del archivo generado
                $domicilio->update([
                    'status' => 'Guía Generada',
                    'invoice_url' => "{$url}/{$fileName}",
                ]);

                // Crear un nuevo estado en la tabla de estados
                \App\Status::create([
                    'status' => 'Guía Generada',
                    'observation' => '',
                    'domicilio_id' => $domicilio->id,
                    'domiciliario' => $domicilio->domiciliario ? $domicilio->domiciliario->name : '',
                ]);

                // Añadir la guía generada al array de resultados
                $guides[] = [
                    'file' => "{$url}/{$fileName}",
                    'domicilio' => $domicilio
                ];
            }

            return [
                'success' => true,
                'data' => $guides
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'code' => 400
            ];
        }
    }



    /**
     * Generate guides label products
     * @param array $products
     */
    public function generateItemLabel($products)
    {
        try {
            $label = '';
            // generate products label
            foreach ($products as $prod => $product) {
                $label .= "{$product->name} X{$product->quantity}";
                if ($prod >= 0 and $prod < (count($products) - 1)) {
                    $label .= ', ';
                }
                if ($prod == (count($products) - 1)) {
                    $label .= '.';
                }
            }

            // return label
            return $label;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * downlaod Excell
     */
    public function exportExcel()
    {
        try {
            return Excel::download(new DomicilioExcel, 'order_excel.xlsx');
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * impor from excel
     * @param array $params
     */
    public function importFromExcel(array $params)
    {
        try {
            $file = $params['file'];
            $filePath = $file->storeAs('temp', 'order_file.xlsx', 'public');
            $import = Excel::import(new DomicilioImport, storage_path('app/public/' . $filePath));
            \Storage::delete($filePath);
            return $import;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * get order items by id
     * @param int $id
     */
    public function getItemsById(int $id)
    {
        try {
            $items = DB::table('domicilio_products')
                ->select(DB::raw("CONCAT(domicilio_products.name, ' x ', domicilio_products.quantity) as item"))
                ->where('domicilio_products.domicilio_id', $id)
                ->get();
            return $items;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Create orders
     * @param array $params
     */
    public function createOrder(array $params)
    {
        DB::beginTransaction();
        try {
            // generate client data
            $clientId = ClientController::saveNewClient($params['client']);
            $client = \App\Client::find($clientId);
            // generate sender data
            $senderData = $this->senderServices->createSender($params['sender']);
            $params['log_sender_id'] = $senderData ? $senderData->id : null;
            // get order price by zone
            $zone = \App\Zone::findOrFail($client->zone_id);
            $params['delivery'] = !is_null($zone->finally_price) ? $zone->finally_price : ENV('ORDER_PRICE');
            // prepare zone and city id in domicilio
            $params['zone_id'] = $zone->id;
            $params['city_id'] = $client->city_id;
            $domicilio = Domicilio::create(array_merge($params, ['status' => 'Pendiente', 'client_id' => $client->id]));
            $numberGuia = Str::padLeft($domicilio->id, 9, 0);
            $reference = env('ONEX_DELIVERY_PREFIX') . "-$numberGuia";
            $domicilio->update(['reference' => $reference]);
            // generate initial status
            $status = \App\Status::create([
                'status' => 'Pendiente',
                'observation' => isset($params['observation']) ? $params['observation'] : '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);

            if (isset($params["dropshipper_information"])) {
                DropshipperSettlement::create([
                    'domicilio_id' => $domicilio->id,
                    'name_dropshipper' => $params["dropshipper_information"]["name_dropshipper"],
                    'provider_cost' => $params["dropshipper_information"]["provider_cost"]
                ]);
            }

            DB::commit();
            return $domicilio;
        } catch (Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Cancelamos la orden
     * @param $reference
     */
    public function cancelOrder($reference)
    {
        try {
            $order = Domicilio::where('reference', $reference)->first();
            if (!$order) {
                return $this->handlerException('No existe la orden que intentas cancelar.');
            }
            if ($order->status === DomicilioStatusConst::DELIVERED) {
                return $this->handlerException(
                    "La orden no puede ser cancelada ya que se encuentra en estado {$order->status}."
                );
            }
            $order->update(['status' => DomicilioStatusConst::CANCELED]);

            $this->onexfyService->notifyChangeStatusOrder(
                $order->reference,
                DomicilioStatusConst::CANCELED,
                "Orden cancelada"
            );

            return $order;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Filtra los estados y las novedades
     * @param $reference
     */
    public function showOrderDetails($reference)
    {
        try {
            $order = Domicilio::where('reference', $reference)
                ->with(['statusDomicilio', 'News'])
                ->first();
            if (!$order) {
                return $this->handlerException('No existe la orden que intentas consultar.');
            }
            return new DomicilioDataResource($order);
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function findOrder($param)
    {
        try {
            $domicilio = Domicilio::where('reference', 'like', "%$param%")
                ->get()
                ->first();
            return new DomicilioResource($domicilio);
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function findOrdersByDomiciliario($domiciliarioId, $perPage = 10, $from = null, $to = null)
    {
        try {

            $domicilios = Domicilio::query()
                ->where('domiciliario_id', $domiciliarioId);

            if (!ParameterAccesorService::domiciliariesCanGetOrdersWithoutAssignation()) {
                $domicilios->orWhere(function ($query) use ($domiciliarioId) {
                    $query->whereNull('domiciliario_id')
                        ->whereHas('zone.city', function ($q) use ($domiciliarioId) {
                            $q->whereHas('domiciliarios', function ($subQuery) use ($domiciliarioId) {
                                $subQuery->where('domiciliarios.id', $domiciliarioId);
                            });
                        });
                });
            }

            if ($from && $to) {
                $domicilios->whereBetween('created_at', [$from, $to]);
            }

            $domicilios = $domicilios->orderBy('id', 'DESC')->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => DomicilioResource::collection($domicilios),
                'pagination' => [
                    'total' => $domicilios->total(),
                    'per_page' => $domicilios->perPage(),
                    'current_page' => $domicilios->currentPage(),
                    'last_page' => $domicilios->lastPage(),
                    'next_page_url' => $domicilios->nextPageUrl(),
                    'prev_page_url' => $domicilios->previousPageUrl(),
                ],
            ]);
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function deleteDomicilio($domicilio)
    {
        try {
            $this->setStatuses([$domicilio->id], DomicilioStatusConst::CANCELED);
            return $domicilio;
        } catch (Exception $e) {
            return $this->handlerException($e->getMessage(), 500);
        }
    }
}
