<?php

namespace App\Http\Requests\Domiciliario;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Domiciliario;

class PutRequest extends FormRequest
{
    public function authorize()
    {
        return true;
    }

    public function rules()
    {
        $domiciliarioId = $this->route('domiciliario');

        return [
            'name' => [
                'required',
                'min:3',
                'max:50',
                'regex:/^[A-Za-zÁÉÍÓÚÜáéíóúüñÑ\s]+$/',
            ],
            'email' => [
                'required',
                'email',
                Rule::unique('domiciliarios', 'email')->ignore($domiciliarioId),
            ],
            'dni' => [
                'required',
                'max:15',
                'min:8',
                Rule::unique('domiciliarios', 'dni')->ignore($domiciliarioId, 'id'),
            ],
            'phone' => 'required|max:15',
            'hiring_method' => 'required',
            'password' => [
                'nullable',
                'min:8',
                'max:12',
                'regex:/(?=.*\d)(?=.*[a-z])(?=.*[A-Z])/',
                'not_regex:/(.)\1{2,}/',
            ],
            'file' => 'image|mimes:jpg,png|max:2048',
            'plate' => 'required|max:10',
            'city' => 'required|numeric',
        ];
    }

    public function messages()
    {
        return [
            'name.required' => 'Campo nombre requerido.',
            'name.min' => 'El nombre no debe tener menos de 3 caracteres.',
            'name.max' => 'El nombre no debe tener más de 50 caracteres.',
            'name.regex' => 'El nombre debe contener solo letras y espacios.',
            'email.required' => 'El campo correo electrónico es obligatorio.',
            'email.email' => 'Correo electrónico no válido.',
            'email.unique' => 'El correo electrónico ingresado ya está en uso.',
            'dni.required' => 'El campo DNI es obligatorio.',
            'dni.unique' => 'El DNI ingresado ya está en uso.',
            'dni.max' => 'El DNI no debe tener más de 10 caracteres.',
            'phone.required' => 'Campo teléfono requerido.',
            'hiring_method.required' => 'Campo método de contratación requerido.',
            'file.image' => 'El archivo debe ser una imagen.',
            'file.mimes' => 'El archivo debe estar en formato JPG o PNG.',
            'file.max' => 'El archivo no debe pesar más de 2 MB.',
            'city.required' => "La ciudad donde opera el domiciliario es requerida",
            'city.numeric' => "La ciudad debe ser un número",
        ];
    }
}
