<?php

namespace App\Http\Controllers;

use App\Status;
use App\Domicilio;
use Illuminate\Http\Request;
use App\Helpers\Helper;
use App\Http\Controllers\AuditController;
use App\Http\Resources\Domicilio\DomicilioResource;
use App\Http\Resources\Domicilio\DomicilioCollection;
use Illuminate\Support\Facades\Event;
use App\Events\Status\Change;
use App\Http\Requests\Status\StatusIndex;
use App\Http\Requests\Status\VerifyMassiveRequest;
use App\Services\Status\StatusService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Traits\ApiResponser;

class StatusController extends Controller
{
    use ApiResponser;
    protected $statusService;

    public function __construct(StatusService $statusService)
    {
        $this->statusService = $statusService;
    }

    public function index(StatusIndex $request)
    {
        try {
            $statuses = $this->statusService->getStatusList($request->user_id);
            return Helper::response("data", $statuses, 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $mainId = $request->user()->getIdMain($request->user());
            // generamos los estados de las ordenes
            $status = Status::create([
                'status' => $request->status,
                'observation' => $request->observation,
                'domiciliario' => $request->user,
                'domicilio_id' => $request->domicilio,
            ]);
            $domiciliario = $request->domiciliario ? $request->domiciliario : null;
            $status->domicilio->update(['status' => $request->status, 'domiciliario_id' => $domiciliario]);
            if ($request->status === 3)
                $status->domicilio->update(['approve' => true, 'date_pay' => Carbon::now()->format('Y-m-d'), 'total_payment' => $status->domicilio->total, 'is_pay' => 'Si']);
            //save auditoria
            AuditController::store($request->user()->name, 'Ha modificado el estado del domicilio con la referencia: ' . $status->domicilio->reference, 'Estados', $mainId);
            $sendEmail = Event::dispatch(new Change($status));
            // preparo la data a retornar.
            $domicilios = Domicilio::Desc()->UserId($mainId)->Now();
            return Helper::response('success', ['domicilio' => new DomicilioResource($status->domicilio), 'allDomicilios' => new DomicilioCollection($domicilios)], 201);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }


    public function massiveVerify(VerifyMassiveRequest $request)
    {
        try {
            DB::beginTransaction();

            $domicilios = $request["items"];
            $result = $this->statusService->verifyDomicilios($domicilios);
            DB::commit();

            return $this->successResponse([
                "message" => "Se ha verificado los domicilios correctamente",
                "data" => $result
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return Helper::response('error', $e->getMessage(), 500);
        }
    }
}
