<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use App\User;
use App\LogSender;
use App\Traits\ApiResponser;
use Illuminate\Http\Request;
use App\Http\Requests\Sender\LogSenderRequest;
use App\Services\LogSender\LogSenderServices;
use App\Http\Resources\Sender\SenderResource;
use App\Http\Resources\Sender\SenderCollection;
use App\Http\Middleware\PermissionsMiddleware;
use Illuminate\Support\Facades\Log;

class LogisticaRemitenteController extends Controller
{
    use ApiResponser;

    public $service;

    /**
     * construct
     */
    public function __construct()
    {
        $this->service = new LogSenderServices();
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('sender-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('sender-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('sender-update'))->handle($request, $next);
        })->only(['update']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('sender-delete'))->handle($request, $next);
        })->only(['destroy']);
    }

    public function index(Request $request)
    {
        try {
            // params
            $page = $request->page ? $request->page : 1;
            $search = $request->search ? $request->search : '';
            $perPage = $request->perPage ? $request->perPage : 10;
            $filters = $request->filters ? $request->filters : false;
            // return data
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->listSenders($page, $perPage, $search, $filters)
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    public function store(LogSenderRequest $request)
    {
        DB::beginTransaction();
        try {
            // store data;
            $data = $this->service->storeSender($request->all());
            // return
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha creado el remitente correctamente.',
                'data' => $data
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }


    public function show($id)
    {
        try {
            $data = LogSender::where('id', $id)->first();
            return $this->successResponse(new SenderResource($data));
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    public function update(LogSenderRequest $request)
    {
        DB::commit();
        try {
            Log::info("EJECUTANDO METODO UPDATE LogisticalRemittenceController");
            Log::info(json_encode($request->user()));
            // set data
            $data = $this->service->updateSender($request->all());
            // return data
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'data' => $data,
                'message' => 'Se ha modificado el remitente correctamente.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            // delete sender
            $data = $this->service->deleteSender($id);
            // return data
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha eliminado el remitente correctamente.',
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * get for select
     */
    public function getForSelect(Request $request)
    {
        try {
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getForSelect(),
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
