<?php

namespace App\Http\Controllers;

use App\Domiciliario;
use Illuminate\Http\Request;
use App\Helpers\Helper;
use App\Http\Controllers\AuditController;
use App\Http\Resources\Domiciliario\DomiciliarioResource;
use App\Http\Resources\Domiciliario\DomiciliarioCollection;
use App\Http\Requests\Domiciliario\PostRequest;
use App\Http\Requests\Domiciliario\PutRequest;
use App\Traits\ApiResponser;
use Illuminate\Support\Facades\DB;
use App\Http\Middleware\PermissionsMiddleware;
use App\Services\Domiciliario\DomiciliarioService;

class DomiciliarioController extends Controller
{
    use ApiResponser;

    protected $domiciliarioService;

    public function __construct(DomiciliarioService $domiciliarioService)
    {
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-update'))->handle($request, $next);
        })->only(['update']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-delete'))->handle($request, $next);
        })->only(['destroy']);

        $this->domiciliarioService = $domiciliarioService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            // saco el id principal de la tabla recursiva usuarios.
            $domiciliarios = Domiciliario::all();
            return Helper::response('success', new DomiciliarioCollection($domiciliarios), 200);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    // Raul
    public function search(Request $request)
    {
        try {
            $domiciliarios = DB::table('domiciliarios')
                ->where('name', 'like', "%{$request->name}%")->get();
            return Helper::response('success', $domiciliarios, 200);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    public function store(PostRequest $request)
    {
        DB::beginTransaction();
        try {
            $domiciliario = $this->domiciliarioService->createDomiciliario($request);

            DB::commit();

            return Helper::response('success', [
                'message' => 'Se ha creado el domiciliario correctamente.',
                'data' => new DomiciliarioResource($domiciliario)
            ], 201);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function show(Domiciliario $domiciliario)
    {
        //
        try {
            return $this->successResponse(new DomiciliarioResource($domiciliario));
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    public function update(PutRequest $request, Domiciliario $domiciliario)
    {
        try {
            DB::beginTransaction();

            $mainId = $request->user()->getIdMain($request->user());

            $this->domiciliarioService->updateDomiciliario($request, $domiciliario);

            AuditController::store($request->user()->name, 'Ha modificado al domiciliario: ' . $domiciliario->name, 'Domiciliario', $mainId);
            DB::commit();
            return Helper::response('success', ['message' => 'Se ha modificado el domiciliario correctamente', 'data' => new DomiciliarioResource($domiciliario)], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Domiciliario $domiciliario)
    {
        try {
            $mainId = $request->user()->getIdMain($request->user());
            AuditController::store($request->user()->name, 'Ha eliminado al domiciliario: ' . $domiciliario->name, 'Domiciliario', $mainId);
            $domiciliario->delete();
            //delete user from bbdd
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            DB::table('users')->where('domiciliario_id', '=', $domiciliario->id)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            return Helper::response('success', ['message' => 'Se ha eliminado el domiciliario correctamente', 'data' => new DomiciliarioResource($domiciliario)], 201);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }
}
