<?php

namespace App\Http\Controllers;

set_time_limit(300000000);

use App\DispatchRelation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\ApiResponser;
use App\Http\Requests\DispatchRelation\Store;
use PDF;
use App\DispatchRelationItem;
use Carbon\Carbon;
use App\Services\Relations\RelationDispathService;
use App\Http\Middleware\PermissionsMiddleware;
use App\Http\Requests\DispatchRelation\DispatchRelationCloseRequest;
use App\Http\Requests\DispatchRelation\AddOrderDispatchRequest;
use App\Http\Requests\DispatchRelation\deleteOrderDispatchRequest;

class DispatchRelationController extends Controller
{
    use ApiResponser;

    public $service;

    /**
     * constructor
     */
    public function __construct(RelationDispathService $relationDispathService)
    {
        $this->service = $relationDispathService;
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-delete'))->handle($request, $next);
        })->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            // get query params
            $perPage = $request->perPage ? $request->perPage : 10;
            $from = $request->from ? $request->from : Carbon::now()->startOfMonth();
            $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfMonth();
            $search = $request->search ? $request->search : '';
            // get query
            $dispatchRelations = $this->service->listRelation($perPage, $from, $to, $search);
            // return data
            return $this->successResponse([
                'success' => true,
                'data' => $dispatchRelations
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Store $request)
    {
        //
        DB::begintransaction();
        try {
            $dispatchRelation = $this->service->registerRelationDispatch($request);
            // retornamos la data
            DB::commit();
            return $this->successCreatedResponse([
                'data' => $dispatchRelation,
                'message' => 'Relación de despacho creada correctamente.',
                'pdf' => $this->service->generatePDF($dispatchRelation->id)
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\DispatchRelation  $dispatchRelation
     * @return \Illuminate\Http\Response
     */
    public function show(DispatchRelation $dispatchRelation)
    {
        //
        try {
            return $dispatchRelation;
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\DispatchRelation  $dispatchRelation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, DispatchRelation $dispatchRelation)
    {
        //
        DB::begintransaction();
        try {
            $dispatchRelation->update(['status' => 'Completada']);
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha completado la relación de despacho',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\DispatchRelation  $dispatchRelation
     * @return \Illuminate\Http\Response
     */
    public function destroy(DispatchRelation $dispatchRelation)
    {
        //
        DB::begintransaction();
        try {
            $dispatchRelation->delete();
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha eliminado la relación de despacho',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * print pdf
     *
     */
    public function emitPdf(Request $request)
    {
        try {
            if (!$request->has('id'))
                return $this->errorResponse('Debes enviar el id de la relación a generar.', 400);
            return $this->service->generatePDF($request->id);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * List relation by domiciliario
     *
     * @return \Illuminate\Http\Response
     */
    public function getTodayRelation(Request $request)
    {
        //
        try {
            $start = Carbon::now()->format('Y-m-d') . ' 00:00:00';
            $end = Carbon::now()->format('Y-m-d') . ' 23:59:59';
            $dispatchRelations = DispatchRelation::select(
                'dispatch_relations.*',
                //'log_senders.name as sender_name',
                'domiciliarios.name as domiciliario_name',
                // DB::raw("COUNT(dispatch_relation_items.id) as totalItems")
            )
                //->leftJoin('log_senders', 'log_senders.id', '=', 'dispatch_relations.log_sender_id')
                ->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'dispatch_relations.domiciliario_id')
                ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.dispatch_relation_id', '=', 'dispatch_relations.id')
                ->where('dispatch_relations.domiciliario_id', $request->domiciliarioId)
                ->whereBetween('dispatch_relations.created_at', [$start, $end])
                ->groupBy('dispatch_relations.id')
                ->get();

            // Recorrer cada relación de despacho para calcular estadísticas - HERC
            foreach ($dispatchRelations as $dispatchRelation) {
                // Calcular el total de domicilios realizados. - HERC
                $dispatchRelation->totalItems = DB::table('dispatch_relation_items')
                    ->where('dispatch_relation_id', $dispatchRelation->id)
                    ->count();

                // Calcular la cantidad domicilios entregados. - HERC
                $dispatchRelation->delivered = DB::table('dispatch_relation_items')
                    ->join('domicilios', 'dispatch_relation_items.domicilio_id', '=', 'domicilios.id')
                    ->where('dispatch_relation_id', $dispatchRelation->id)
                    ->where('domicilios.status', 'Entregado')
                    ->count();

                // Obtener el valor del campo "total" de la tabla "domicilios" y se usa para mostrar el recaudo total. - HERC
                $dispatchRelation->recaudo = DB::table('domicilios')
                    ->join('dispatch_relation_items', 'domicilios.id', '=', 'dispatch_relation_items.domicilio_id')
                    ->where('dispatch_relation_items.dispatch_relation_id', $dispatchRelation->id)
                    ->sum('domicilios.total');
            }

            return $this->successResponse([
                'success' => true,
                'data' => $dispatchRelations
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * List relations by domiciliario and do some filters
     * @param Request $request
     */
    public function listByDomiciliario(Request $request)
    {
        try {
            // get filter data
            $start = $request->from ? $request->from : Carbon::now()->startOfWeek()->format('Y-m-d');
            $end = $request->to ? $request->to : Carbon::now()->endOfWeek()->format('Y-m-d');
            $page = $request->page ? $request->page : 1;
            $perPage = $request->perPage ? $request->perPage : 12;
            $domiciliarioId = $request->domiciliarioId;
            // do query and return data
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getRelationList(
                    $start . '00:00:00',
                    $end . ' 23:59:59',
                    $page,
                    $perPage,
                    $domiciliarioId
                )
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * List order by relation dispatch
     * @param Request $request
     */
    public function loadOrderByRelationOrNot(Request $request)
    {
        try {
            $start = $request->from ? $request->from : Carbon::now()->startOfMonth()->format('Y-m-d');
            $end = $request->to ? $request->to : Carbon::now()->endOfMonth()->format('Y-m-d');
            $page = $request->page ? (int)$request->page : 0;
            $perPage = $request->perPage ? (int)$request->perPage : 12;
            $relationId = $request->relationId ? (int)$request->relationId : null;
            $domiciliarioId = (int)$request->domiciliarioId; // Conversión a entero
            $status = $request->status ? $request->status : 'todos';

            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getOrderByRelationOrNot(
                    $start . ' 00:00:00',
                    $end . ' 23:59:59',
                    $page,
                    $perPage,
                    $relationId,
                    $domiciliarioId, // Aseguramos que esto sea un entero
                    $status
                )
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    public function getOrdersByStatusTab(Request $request)
    {
        try {
            $start = $request->from ? $request->from : Carbon::now()->startOfMonth()->format('Y-m-d');
            $end = $request->to ? $request->to : Carbon::now()->endOfMonth()->format('Y-m-d');
            $page = $request->page ? (int)$request->page : 0;
            $perPage = $request->perPage ? (int)$request->perPage : 12;
            $relationId = $request->relationId ? (int)$request->relationId : null;
            $domiciliarioId = (int)$request->domiciliarioId; // Conversión a entero
            $status = $request->status ? $request->status : 'todos';

            $orders = $this->service->getOrdersByRelationAndStatus(
                $start . ' 00:00:00',
                $end . ' 23:59:59',
                $page,
                $perPage,
                $relationId,
                $domiciliarioId, // Aseguramos que esto sea un entero
                $status
            );

            return $this->successResponse([
                'success' => true,
                'data' => $orders
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Filter order by id
     * @param $id
     * @param Request $request
     */
    public function getOrderById($id, Request $request)
    {
        try {
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getOrderById($id)
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Filter order by id
     * @param $id
     * @param Request $request
     */
    public function getOrderByReference($id, Request $request)
    {
        try {
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getOrderByReference($id)
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * mark as compelte the relation dispatch
     */
    public function completeRelation($id)
    {
        //
        DB::begintransaction();
        try {
            $relation = $this->service->completeRelation($id);
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha completado la relación de despacho',
                'data' => $relation,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function generateRelationClose(DispatchRelationCloseRequest $request)
    {
        DB::beginTransaction();
        try {
            $data = $request->all();
            $url = $this->service->generateRelationClose($data["relation"], $data["items"]);
            DB::commit();
            return $url;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function addOrderInDispatchRelationship(AddOrderDispatchRequest $request)
    {
        DB::beginTransaction();
        try {
            $data = $request->all();
            $result = $this->service->addDomicilioToDispatch($data["relation_id"], $data["reference"]);

            DB::commit();
            return $this->successResponse([
                'success' => true,
                'data' => $result
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function deleteOrderInDispatchRelationship($id)
    {
        DB::beginTransaction();
        try {
            $result = $this->service->deleteOrderInDispatch($id);

            DB::commit();
            return $this->successResponse([
                'success' => true,
                'data' => $result
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
