<?php

namespace App\Http\Controllers;

use App\Client;
use Illuminate\Http\Request;
use App\Helpers\Helper;
use App\Http\Resources\Client\ClientShowResource;
use App\Http\Requests\Client\PostRequest;
use App\Http\Requests\Client\PutRequest;
use App\Services\Client\ClientServices;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ClientsImport;
use App\Http\Resources\Client\ClientCollection;
use App\Traits\ApiResponser;

class ClientController extends Controller
{
    use ApiResponser;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    protected $clientService;

    public function __construct(ClientServices $clientService)
    {
        $this->clientService = $clientService;
    }

    public function index(Request $request)
    {
        try {
            $perPage = $request->perPage ? $request->perPage : 10;
            $search = $request->search ? $request->search : '';

            return $this->clientService->getAllClients($perPage, $search);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    public function getListClient(Request $request)
    {
        try {
            $clients = $this->clientService->getClientList($request);
            return [
                'success' => true,
                'message' => 'Listado de clientes.',
                'data' =>  new ClientCollection($clients),
            ];
        } catch (\Exception $e) {
            return Helper::response("error", $e->getMessage(), 500);
        }
    }

    /**
     * Display a listing of the resource by user_id.
     *
     * @return \Illuminate\Http\Response
     */
    public function clientsByUserId($user_id)
    {
        try {
            $clients = Client::where('user_id', $user_id)->orderby('id', 'DESC')->get();
            return [
                'success' => true,
                'message' => 'Listado de clientes.',
                'data' =>  new ClientCollection($clients),
            ];
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(PostRequest $request)
    {
        try {
            $client = $this->clientService->registerClient($request);
            return Helper::response('success', ['meesage' => 'Se ha creado el cliente correctamente', 'client' => $client], 201);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Attribute  $attribute
     * @return \Illuminate\Http\Response
     */
    public function show(Client $client)
    {
        //
        try {
            return Helper::response('success', new ClientShowResource($client), 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Client  $client
     * @return \Illuminate\Http\Response
     */
    public function update(PutRequest $request, Client $client)
    {
        try {
            $client = $this->clientService->updateClient($request, $client);
            return Helper::response('success', 'Se ha editado el cliente', 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Client  $client
     * @return \Illuminate\Http\Response
     */
    public function destroy(Client $client)
    {
        //delete client
        try {
            $client->delete();
            return Helper::response('success', 'Se ha eliminado el cliente correctamente', 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    // create new client in our bbdd
    static function saveNewClient($client)
    {
        //
        $validateIfIsset = Client::Phone($client['phone'])->first();
        if (!isset($validateIfIsset)) {
            $client = Client::create($client);
            return $client->id;
        }
        $validateIfIsset->fill($client);
        $validateIfIsset->save();
        return $validateIfIsset->id;
    }

    static function saveNewClient2($client, $mainId)
    {
        //
        $validateIfIsset = Client::Dni($client['dni'])->where('user_id', $mainId)->first();
        if (!isset($validateIfIsset)) {
            $client = Client::create(array_merge($client, ['user_id' => $mainId, 'document_type_id' => 3]));
            return $client;
        }
        return $validateIfIsset;
    }

    public function filter(Request $request, $dni, ClientServices $clientServices)
    {
        //
        try {
            // saco el id principal del usuario
            $client = Client::where('dni', $dni)->first();
            if ($client) {
                $client->purse = $client->Purse;
                return Helper::response('success', $client, 200);
            } else
                return $this->errorResponse('No existe un cliente con esta cédula, se procedera a crear.', 404);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    public function importClients(Request $request, ClientServices $clientServices)
    {
        try {
            $mainId = $request->user()->getIdMain($request->user());
            Excel::import(new ClientsImport($mainId), $request->file('file')->store('temp'));
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }
}
