<?php

namespace App\Services\DomicilioNews;

use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Domicilio;
use App\DomicilioNews;
use App\Traits\ApiResponser;
use App\Services\Domicilio\DomicilioServices;
use App\Enums\NewsConst;
use App\Enums\DomicilioStatusConst;
use App\CoordinatorHasCity;

/**
 *
 */
class DomicilioNewServices
{
    use ApiResponser;

    protected $domicilioServices;

    function __construct(DomicilioServices $domicilioService)
    {
        $this->domicilioServices = $domicilioService;
    }

    public function getAllDomiciliosWithNews(Request $request)
    {
        try {
            $perPage = isset($request->per_page) ? $request->per_page : (isset($request->rowsPerPage) ? $request->rowsPerPage : 100);
            $from = $request->from ? $request->from . ' 00:00:00' : Carbon::now()->startOfYear();
            $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfYear();
            $clientId = $request->client_id ?? null;

            // select query
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw(
                    "(SELECT domicilios_reschedulings.rescheduling_date FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_date"
                ),
                DB::raw(
                    "(SELECT domicilios_reschedulings.rescheduling_start_hour FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_start_hour"
                ),
                DB::raw(
                    "(SELECT domicilios_reschedulings.rescheduling_end_hour FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_end_hour"
                ),
                DB::raw(
                    "(SELECT domicilios_reschedulings.description FROM domicilios_reschedulings WHERE domicilios.id = domicilios_reschedulings.domicilio_id ORDER BY domicilios_reschedulings.rescheduling_date DESC LIMIT 1) as rescheduling_description"
                ),
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id LIMIT 1) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id LIMIT 1) as zone"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id LIMIT 1) as client"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT SUM(value) FROM domicilio_collections WHERE domicilio_collections.domicilio_id = domicilios.id) as collection"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")
            ])
                ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
                ->where('domicilios.log_sender_id', 'LIKE', '%' . $request->sender_id . '%');

            if (is_null($clientId) && is_null($request->search)) {
                $domicilios->whereBetween('domicilios.created_at', [$from, $to]);
            }
            // Filtro por client_id
            if (!is_null($clientId)) {
                $domicilios->where('domicilios.client_id', '=', $clientId);
            }

            if (!is_null($request->domiciliario_id)) {
                $domicilios->where('domicilios.domiciliario_id', 'LIKE', '%' . $request->domiciliario_id . '%');
            }
            if (!is_null($request->liquidated)) {
                $domicilios->where('domicilios.liquidated', 'LIKE', '%' . $request->liquidated . '%');
            }
            if (!is_null($request->city_id)) {
                $domicilios->where('domicilios.city_id', '=', $request->city_id);
            }
            if (!is_null($request->zone_id)) {
                $domicilios->where('domicilios.zone_id', '=', $request->zone_id);
            }

            $domicilios->where('domicilios.status', '=', DomicilioStatusConst::NEW_EVENT);
            $domicilios->with('news.type');

            $columns = [
                'service',
                'client',
                'domiciliario',
                'id',
                'reference',
                'weight',
                'status',
                'sender',
                'client_city',
                'client_zone',
                'rescheduling_date'
            ];

            if (isset($request->search)) {
                $domicilios = $domicilios->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $request->search . "%'");
            }

            if ($request->user() && is_numeric($request->user()->domiciliario_id)) {
                $domicilios = $domicilios->where('domiciliario_id', $request->user()->domiciliario_id);
            }
            if ($request->user() && is_numeric($request->user()->log_sender_id)) {
                $domicilios = $domicilios->where('log_sender_id', $request->user()->log_sender_id);
            }
            if ($request->user() && $request->user()["role_id"] == 3) {
                $user = $request->user();
                $cityIds = CoordinatorHasCity::where('user_id', $user->id)->pluck('city_id');
                $domicilios->whereIn('domicilios.city_id', $cityIds);
            }

            if (!is_null($request->status)) {
                $domicilios->orderBy('domicilios.news.created_at', 'DESC');
            }

            $domicilios = $domicilios->paginate($perPage);
            return $domicilios;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Create domicilio collection
     * @param array $params
     */
    public function createDomicilioNews(array $params)
    {
        try {
            // create news
            // $domicilioCollection = DomicilioNews::create($params);
            $domicilioCollection = DomicilioNews::create([
                'description' => $params["description"],
                'domicilio_id' => $params["domicilio_id"],
                'new_type_id' => $params["type"]
            ]);

            if (isset($params['file'])) {
                $this->domicilioServices->uploadEvidence(
                    $params['file'],
                    $params['domicilio_id'],
                    'Novedad'
                );
            }
            $domicilio = Domicilio::findOrFail($params['domicilio_id']);
            if ($params["type"] === NewsConst::CANCELLED) {
                $domicilio->update(['status' => DomicilioStatusConst::CANCELED]);
                $this->domicilioServices->setStatuses(
                    array($params['domicilio_id']),
                    DomicilioStatusConst::CANCELED,
                    $domicilioCollection->type ? $domicilioCollection->type->name : '',
                );
            } else {
                $domicilio->update(['status' => 'Novedad']);
                $this->domicilioServices->setStatuses(
                    array($params['domicilio_id']),
                    'Novedad',
                    $domicilioCollection->type ? $domicilioCollection->type->name : '',
                );
            }
            return $domicilioCollection;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    public function getDomicilioNewById($domicilioId)
    {
        try {
            return DomicilioNews::where("domicilio_id", $domicilioId)->first();
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }
}
